// by Dennis Yurichev

// portions of code was shamelessly stolen from GNU GSL library and reworked.
// https://www.gnu.org/software/gsl/

#include <time.h>
#include <stdbool.h>
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <string>
#include <set>
#include <map>
#include <vector>

#define DBG

//#define MINIMIZE
#define MAXIMIZE

// From https://github.com/ESultanik/mtwister

/* An implementation of the MT19937 Algorithm for the Mersenne Twister
 * by Evan Sultanik.  Based upon the pseudocode in: M. Matsumoto and
 * T. Nishimura, "Mersenne Twister: A 623-dimensionally
 * equidistributed uniform pseudorandom number generator," ACM
 * Transactions on Modeling and Computer Simulation Vol. 8, No. 1,
 * January pp.3-30 1998.
 *
 * http://www.sultanik.com/Mersenne_twister
 */

#define STATE_VECTOR_LENGTH 624
#define STATE_VECTOR_M      397 /* changes to STATE_VECTOR_LENGTH also require changes to this */

typedef struct tagMTRand {
        unsigned long mt[STATE_VECTOR_LENGTH];
        int index;
} MTRand;

MTRand global;

#define UPPER_MASK		0x80000000
#define LOWER_MASK		0x7fffffff
#define TEMPERING_MASK_B	0x9d2c5680
#define TEMPERING_MASK_C	0xefc60000

void m_seedRand(MTRand* rand, unsigned long seed) {
        /* set initial seeds to mt[STATE_VECTOR_LENGTH] using the generator
         * from Line 25 of Table 1 in: Donald Knuth, "The Art of Computer
         * Programming," Vol. 2 (2nd Ed.) pp.102.
         */
        rand->mt[0] = seed & 0xffffffff;
        for(rand->index=1; rand->index<STATE_VECTOR_LENGTH; rand->index++) {
                rand->mt[rand->index] = (6069 * rand->mt[rand->index-1]) & 0xffffffff;
        }
}

/**
 * Creates a new random number generator from a given seed.
 */
MTRand seedRand(unsigned long seed) {
        MTRand rand;
        m_seedRand(&rand, seed);
        return rand;
}

/**
 * Generates a pseudo-randomly generated long.
 */
unsigned long genRandLong(MTRand* rand) {

        unsigned long y;
        static unsigned long mag[2] = {0x0, 0x9908b0df}; /* mag[x] = x * 0x9908b0df for x = 0,1 */
        if(rand->index >= STATE_VECTOR_LENGTH || rand->index < 0) {
                /* generate STATE_VECTOR_LENGTH words at a time */
                int kk;
                if(rand->index >= STATE_VECTOR_LENGTH+1 || rand->index < 0) {
                        m_seedRand(rand, 4357);
                }
                for(kk=0; kk<STATE_VECTOR_LENGTH-STATE_VECTOR_M; kk++) {
                        y = (rand->mt[kk] & UPPER_MASK) | (rand->mt[kk+1] & LOWER_MASK);
                        rand->mt[kk] = rand->mt[kk+STATE_VECTOR_M] ^ (y >> 1) ^ mag[y & 0x1];
                }
                for(; kk<STATE_VECTOR_LENGTH-1; kk++) {
                        y = (rand->mt[kk] & UPPER_MASK) | (rand->mt[kk+1] & LOWER_MASK);
                        rand->mt[kk] = rand->mt[kk+(STATE_VECTOR_M-STATE_VECTOR_LENGTH)] ^ (y >> 1) ^ mag[y & 0x1];
                }
                y = (rand->mt[STATE_VECTOR_LENGTH-1] & UPPER_MASK) | (rand->mt[0] & LOWER_MASK);
                rand->mt[STATE_VECTOR_LENGTH-1] = rand->mt[STATE_VECTOR_M-1] ^ (y >> 1) ^ mag[y & 0x1];
                rand->index = 0;
        }
        y = rand->mt[rand->index++];
        y ^= (y >> 11);
        y ^= (y << 7) & TEMPERING_MASK_B;
        y ^= (y << 15) & TEMPERING_MASK_C;
        y ^= (y >> 18);
        return y;
}

/**
 * Generates a pseudo-randomly generated double in the range [0..1).
 */
double genRand(MTRand* rand) {
        return((double)genRandLong(rand) / (unsigned long)0xffffffff);
}

// [begin, end)
int rand_reg (MTRand* rand, int begin, int end)
{
        if (end==begin)
                return end;
        assert (end>begin);
        int rt=(genRandLong(rand) % (end-begin))+begin;
        assert (rt>=begin);
        assert (rt<end);
        return rt;
};

// **************************************

int next_vertex_n=0;
int vertex_t=0;
std::vector<std::string> vertex_aliases;
std::map<std::string, int> vertex_numbers;
std::map<int, std::set<int> > edges;
//int state[vertex_t]; // permutation is here. position of each vertex

int fitness_fn(void* state_in, bool print)
{
        int* state=(int*)state_in;
        int fwd=0, back=0;
        // enum all edges. find src_n and dst_n
        for (auto e : edges)
        {
                int src_n=e.first;
                assert (src_n < vertex_t);
                for (int dst_n : edges[src_n])
                {
                        assert (dst_n < vertex_t);
                        if (state[src_n] < state[dst_n])
                                fwd+=1;
                        else
                                back+=1;
                };
        }
        if (print)
        {
                printf ("; %s() fwd= %d\n", __FUNCTION__, fwd);
                printf ("; %s() back=%d\n", __FUNCTION__, back);
        }
        return fwd;
};

// this code is from GNU GSL:

typedef struct
{
        int iters_fixed_T;    /* how many iterations at each temperature? */
        /* the following parameters are for the Boltzmann distribution */
        double k, t_initial, mu_t, t_min;
} gsl_siman_params_t;

#define GSL_LOG_DBL_MIN   (-7.0839641853226408e+02)

double boltzmann(double E, double new_E, double T, gsl_siman_params_t *params)
{
        double x = -(new_E - E) / (params->k * T);
        /* avoid underflow errors for large uphill steps */
        return (x < GSL_LOG_DBL_MIN) ? 0.0 : exp(x);
}

typedef double (*gsl_siman_Efunc_t) (void *xp);
typedef void (*gsl_siman_step_t) (void *xp);

void gsl_siman_solve (void *x0_p, gsl_siman_Efunc_t Ef,
                gsl_siman_step_t take_step,
                size_t element_size,
                gsl_siman_params_t params)
{
        void *x, *new_x, *best_x;
        double E, new_E, best_E;
        int i;
        double T, T_factor;
        int n_evals = 1, n_iter = 0, n_accepts, n_rejects, n_eless;

        E = Ef(x0_p);

        x = (void *) malloc (element_size);
        memcpy (x, x0_p, element_size);
        new_x = (void *) malloc (element_size);
        best_x =  (void *) malloc (element_size);
        memcpy (best_x, x0_p, element_size);

        best_E = E;

        T = params.t_initial;
        T_factor = 1.0 / params.mu_t;

#ifdef DBG
        fprintf (stderr, "#-iter  #-evals   temperature     position   energy\n");
#endif

        while (1)
        {
                n_accepts = 0;
                n_rejects = 0;
                n_eless = 0;

                for (i = 0; i < params.iters_fixed_T; ++i)
                {
                        memcpy (new_x, x, element_size);

                        take_step (new_x);
                        new_E = Ef (new_x);

                        if(new_E <= best_E)
                        {
                                memcpy (best_x, new_x, element_size);
                                best_E=new_E;
                        }

                        ++n_evals;
                        if (new_E < E)
                        {
                                if (new_E < best_E)
                                {
                                        memcpy (best_x, new_x, element_size);
                                        best_E = new_E;
                                }

                                /* yay! take a step */
                                memcpy (x, new_x, element_size);
                                E = new_E;
                                ++n_eless;

                        } else
                                if (genRand(&global) < boltzmann(E, new_E, T, &params))
                                {
                                        /* yay! take a step */
                                        memcpy (x, new_x, element_size);
                                        E = new_E;
                                        ++n_accepts;

                                } else {
                                        ++n_rejects;
                                }
                }

#ifdef DBG
                /* see if we need to print stuff as we go */
                /*       printf("%5d %12g %5d %3d %3d %3d", n_iter, T, n_evals, */
                /*           100*n_eless/n_steps, 100*n_accepts/n_steps, */
                /*           100*n_rejects/n_steps); */
                fprintf (stderr, "%5d   %7d  %12g", n_iter, n_evals, T);
                //print_board ((char (*)[8])x);
                //fitness_fn (x);
                fprintf (stderr, "  %12g  %12g\n", E, best_E);
#endif
                /* apply the cooling schedule to the temperature */
                /* FIXME: I should also introduce a cooling schedule for the iters */
                T *= T_factor;
                ++n_iter;
                if (T < params.t_min)
                {
                        break;
                }
        }

        /* at the end, copy the result onto the initial point, so we pass it
           back to the caller */
        memcpy (x0_p, best_x, element_size);

        free (x);
        free (new_x);
        free (best_x);
}

double Ef(void *in)
{
#ifdef MAXIMIZE
        return -(double)fitness_fn (in, false);
#elif defined(MINIMIZE)
        return (double)fitness_fn (in, false);
#else
#error "error"
#endif
};

void take_step(void *in)
{
        int* state=(int*)in;

        int x=rand_reg (&global, 0, vertex_t);
        int y=rand_reg (&global, 0, vertex_t);

        int tmp=state[x];
        state[x]=state[y];
        state[y]=tmp;
};

/* set up parameters for this simulated annealing run */
//#define ITERS_FIXED_T 1000      /* how many iterations for each T? */
#define K 1.0                   /* Boltzmann constant */
#define T_INITIAL 0.008         /* initial temperature */
#define MU_T 1.003              /* damping factor for temperature */
#define T_MIN 2.0e-6

void try_seed(int seed, int iters_fixed_t)
{
        fprintf (stdout, "; Trying seed %d\n", seed);
        global=seedRand(seed);

        int *state=(int*)calloc(vertex_t, sizeof(int));
        for (int i=0; i<vertex_t; i++)
                state[i]=i;

        fprintf (stdout, "; fitness at start\n");
        fitness_fn(state, true);

        //gsl_siman_params_t params = {ITERS_FIXED_T, K, T_INITIAL, MU_T, T_MIN};
        gsl_siman_params_t params = {iters_fixed_t, K, T_INITIAL, MU_T, T_MIN};

#if 1
        gsl_siman_solve (state, Ef,
                        take_step,
                        //sizeof (state),
                        sizeof(int)*vertex_t,
                        params);
#endif
        // print result
        int result[vertex_t];
        for (int i=0; i<vertex_t; i++)
        {
                int position=state[i];
                result[position]=i;
        };
        for (int i=0; i<vertex_t; i++)
                printf ("%s\n", vertex_aliases[result[i]].c_str());

        fprintf (stdout, "; fitness at end:\n");
        fitness_fn(state, true);
};

int find_vertex_n(std::string name)
{
        if (vertex_numbers.find(name)==vertex_numbers.end())
        {
                vertex_numbers[name]=next_vertex_n;
                vertex_aliases.push_back(name);
                edges[next_vertex_n]=std::set<int>();
                next_vertex_n++;
        }
        else
                return vertex_numbers[name];
        return next_vertex_n-1;
};

int main(int argc, char *argv[])
{
        int iters_fixed_t=100;
        if (argc==2)
        {
                iters_fixed_t=atoi (argv[1]);
        };
        fprintf(stdout, "; setting iters_fixed_t to %d\n", iters_fixed_t);

        char buf[1024];
        while (1)
        {
                char* r=fgets(buf, sizeof buf, stdin);
                if (r==NULL)
                        break;
                buf[strlen(buf)-1]=0;

                char *tmp=strchr(buf, ' ');
                if (tmp==NULL)
                {
                        printf ("Fatal Error. Incorrect input: %s\n", buf);
                        printf ("Must be: edge_src edge_dst\n");
                        exit(0);
                };
                *tmp=0;

                std::string src_name=std::string(buf);
                std::string dst_name=std::string(tmp+1);
                int src_n=find_vertex_n(src_name);
                int dst_n=find_vertex_n(dst_name);
                //printf ("adding to edges: %s %s\n", vertex_aliases[src_n].c_str(), vertex_aliases[dst_n].c_str());
                edges[src_n].insert(dst_n);
        }
        vertex_t=next_vertex_n;
#if 0
        // test
        int state[vertex_t];

        state[find_vertex_n("v3")]=0;
        state[find_vertex_n("v5")]=1;
        state[find_vertex_n("v7")]=2;
        state[find_vertex_n("v11")]=3;
        state[find_vertex_n("v8")]=4;
        state[find_vertex_n("v10")]=5;
        state[find_vertex_n("v2")]=6;
        state[find_vertex_n("v9")]=7;

        printf ("%d\n", fitness_fn(state));
        exit(0);
#endif
        //for (int i=0; i<vertex_t; i++)
        //        printf ("vertex_aliases[%d]=%s\n", i, vertex_aliases[i].c_str());

        time_t t_start=time(NULL);
        try_seed(t_start, iters_fixed_t);
        time_t t_end=time(NULL);
        printf ("; seconds spent %ld\n", t_end-t_start);
};

